<?php
require __DIR__ . '/../config.php';

header('Content-Type: application/json');

function json_response(array $data): void
{
    echo json_encode($data);
    exit;
}

$key = $_GET['key'] ?? '';
$hwid = $_GET['hwid'] ?? '';

if ($key === '') {
    json_response(['status' => 'INVALID_KEY']);
}

// Find license by key
$stmt = $mysqli->prepare('SELECT * FROM licenses WHERE license_key = ? LIMIT 1');
$stmt->bind_param('s', $key);
$stmt->execute();
$result = $stmt->get_result();
$license = $result->fetch_assoc();
$stmt->close();

if (!$license) {
    json_response(['status' => 'INVALID_KEY']);
}

// If blocked
if ($license['status'] === 'blocked') {
    json_response(['status' => 'BLOCKED']);
}

// Check expiry
if (!empty($license['expiry_date']) && strtotime($license['expiry_date']) < strtotime(date('Y-m-d'))) {
    // Mark as expired if not already
    if ($license['status'] !== 'expired') {
        $upd = $mysqli->prepare("UPDATE licenses SET status = 'expired' WHERE id = ?");
        $id = (int)$license['id'];
        $upd->bind_param('i', $id);
        $upd->execute();
        $upd->close();
        $license['status'] = 'expired';
    }
    json_response(['status' => 'EXPIRED']);
}

// HWID handling
if (empty($license['hwid'])) {
    // first activation, save HWID (if provided)
    if ($hwid !== '') {
        $upd = $mysqli->prepare('UPDATE licenses SET hwid = ? WHERE id = ?');
        $id = (int)$license['id'];
        $upd->bind_param('si', $hwid, $id);
        $upd->execute();
        $upd->close();
        $license['hwid'] = $hwid;
    }
} else {
    // license already has hwid stored, must match
    if ($hwid === '' || $hwid !== $license['hwid']) {
        json_response(['status' => 'HWID_MISMATCH']);
    }
}

// All good, update last_seen and IP
$ip = $_SERVER['REMOTE_ADDR'] ?? '';
$now = date('Y-m-d H:i:s');

$upd = $mysqli->prepare('UPDATE licenses SET last_seen = ?, ip_address = ? WHERE id = ?');
$id = (int)$license['id'];
$upd->bind_param('ssi', $now, $ip, $id);
$upd->execute();
$upd->close();

// Calculate days left
$daysLeft = null;
if (!empty($license['expiry_date'])) {
    $today = new DateTime('today');
    $expiry = DateTime::createFromFormat('Y-m-d', $license['expiry_date']);
    if ($expiry) {
        $daysLeft = (int)$today->diff($expiry)->format('%r%a');
    }
}

json_response([
    'status' => 'AUTHORIZED',
    'expiry' => $license['expiry_date'],
    'days_left' => $daysLeft,
]);


