<?php
require __DIR__ . '/config.php';
require_admin_login();

// Simple logout
if (isset($_GET['action']) && $_GET['action'] === 'logout') {
    session_destroy();
    header('Location: login.php');
    exit;
}

// Handle row actions: block, allow, extend, delete
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['license_id'], $_POST['do'])) {
    $license_id = (int)$_POST['license_id'];
    $action = $_POST['do'];

    if ($license_id > 0) {
        if ($action === 'block') {
            $stmt = $mysqli->prepare("UPDATE licenses SET status = 'blocked' WHERE id = ?");
            $stmt->bind_param('i', $license_id);
            $stmt->execute();
            $stmt->close();
        } elseif ($action === 'allow') {
            $stmt = $mysqli->prepare("UPDATE licenses SET status = 'active' WHERE id = ?");
            $stmt->bind_param('i', $license_id);
            $stmt->execute();
            $stmt->close();
        } elseif ($action === 'extend') {
            // extend expiry_date by 7 days
            $stmt = $mysqli->prepare("UPDATE licenses SET expiry_date = DATE_ADD(expiry_date, INTERVAL 7 DAY) WHERE id = ?");
            $stmt->bind_param('i', $license_id);
            $stmt->execute();
            $stmt->close();
        } elseif ($action === 'delete') {
            $stmt = $mysqli->prepare("DELETE FROM licenses WHERE id = ?");
            $stmt->bind_param('i', $license_id);
            $stmt->execute();
            $stmt->close();
        }
    }

    header('Location: dashboard.php');
    exit;
}

// Stats
$totals = [
    'total' => 0,
    'active' => 0,
    'blocked' => 0,
    'expired' => 0,
    'online' => 0,
];

// Total
if ($res = $mysqli->query("SELECT COUNT(*) AS c FROM licenses")) {
    $row = $res->fetch_assoc();
    $totals['total'] = (int)$row['c'];
    $res->free();
}

// By status
if ($res = $mysqli->query("SELECT status, COUNT(*) AS c FROM licenses GROUP BY status")) {
    while ($row = $res->fetch_assoc()) {
        $status = $row['status'];
        $cnt = (int)$row['c'];
        if (isset($totals[$status])) {
            $totals[$status] = $cnt;
        }
    }
    $res->free();
}

// Online (last_seen within 2 minutes)
if ($res = $mysqli->query("SELECT COUNT(*) AS c FROM licenses WHERE last_seen IS NOT NULL AND last_seen >= (NOW() - INTERVAL 2 MINUTE)")) {
    $row = $res->fetch_assoc();
    $totals['online'] = (int)$row['c'];
    $res->free();
}

// Fetch all licenses
$licenses = [];
if ($res = $mysqli->query("SELECT * FROM licenses ORDER BY id DESC")) {
    while ($row = $res->fetch_assoc()) {
        $licenses[] = $row;
    }
    $res->free();
}

function days_left_for_license(array $license): ?int
{
    if (empty($license['expiry_date'])) {
        return null;
    }
    $today = new DateTime('today');
    $expiry = DateTime::createFromFormat('Y-m-d', $license['expiry_date']);
    if (!$expiry) {
        return null;
    }
    $diff = (int)$today->diff($expiry)->format('%r%a');
    return $diff;
}

function status_class(string $status): string
{
    if ($status === 'active') {
        return 'status-active';
    }
    if ($status === 'blocked') {
        return 'status-blocked';
    }
    if ($status === 'expired') {
        return 'status-expired';
    }
    return '';
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Dashboard - License Panel</title>
    <link rel="stylesheet" href="styles.css">
</head>
<body>
<div class="container">
    <div class="top-nav">
        <div class="top-nav-title">License Admin Dashboard</div>
        <div class="top-nav-actions">
            <a href="generate.php" class="btn btn-primary">Generate License</a>
            <a href="?action=logout" class="btn btn-outline">Logout</a>
        </div>
    </div>

    <div class="stats-grid">
        <div class="stat-card">
            <div class="stat-label">Total Keys</div>
            <div class="stat-value"><?php echo (int)$totals['total']; ?></div>
        </div>
        <div class="stat-card">
            <div class="stat-label">Active</div>
            <div class="stat-value" style="color:#3fb950;"><?php echo (int)$totals['active']; ?></div>
        </div>
        <div class="stat-card">
            <div class="stat-label">Blocked</div>
            <div class="stat-value" style="color:#ff7b72;"><?php echo (int)$totals['blocked']; ?></div>
        </div>
        <div class="stat-card">
            <div class="stat-label">Expired</div>
            <div class="stat-value" style="color:#e3b341;"><?php echo (int)$totals['expired']; ?></div>
        </div>
        <div class="stat-card">
            <div class="stat-label">Abhi Online (last 2 mins)</div>
            <div class="stat-value" style="color:#3fb950;"><?php echo (int)$totals['online']; ?></div>
        </div>
    </div>

    <div class="card">
        <div class="page-title" style="margin-bottom: 10px;">All Licenses</div>
        <div class="table-wrapper">
            <table>
                <thead>
                <tr>
                    <th>ID</th>
                    <th>Client Name</th>
                    <th>License Key</th>
                    <th>Status</th>
                    <th>Expiry Date</th>
                    <th>Days Left</th>
                    <th>Last Seen</th>
                    <th>Current Activity</th>
                    <th>IP</th>
                    <th>Online</th>
                    <th>Actions</th>
                </tr>
                </thead>
                <tbody>
                <?php if (empty($licenses)): ?>
                    <tr>
                        <td colspan="11" class="muted">No licenses yet.</td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($licenses as $lic):
                        $daysLeft = days_left_for_license($lic);
                        $isOnline = !empty($lic['last_seen']) && (strtotime($lic['last_seen']) >= (time() - 120));
                        ?>
                        <tr>
                            <td><?php echo (int)$lic['id']; ?></td>
                            <td><?php echo h($lic['client_name']); ?></td>
                            <td><?php echo h($lic['license_key']); ?></td>
                            <td>
                                <span class="status-badge <?php echo status_class($lic['status']); ?>">
                                    <?php echo strtoupper(h($lic['status'])); ?>
                                </span>
                            </td>
                            <td><?php echo h($lic['expiry_date']); ?></td>
                            <td>
                                <?php
                                if ($daysLeft === null) {
                                    echo '-';
                                } else {
                                    echo $daysLeft . ' days';
                                }
                                ?>
                            </td>
                            <td><?php echo h($lic['last_seen']); ?></td>
                            <td><?php echo h($lic['current_activity']); ?></td>
                            <td><?php echo h($lic['ip_address']); ?></td>
                            <td>
                                <?php if ($isOnline): ?>
                                    <span class="badge-online"><span class="badge-dot"></span>Online</span>
                                <?php else: ?>
                                    <span class="muted small">-</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <form method="post" style="display:flex;gap:4px;flex-wrap:wrap;">
                                    <input type="hidden" name="license_id" value="<?php echo (int)$lic['id']; ?>">
                                    <button class="btn btn-danger small" name="do" value="block" type="submit">BLOCK</button>
                                    <button class="btn btn-success small" name="do" value="allow" type="submit">ALLOW</button>
                                    <button class="btn btn-warning small" name="do" value="extend" type="submit">EXTEND</button>
                                    <button class="btn btn-outline small" name="do" value="delete" type="submit" onclick="return confirm('Delete this license?');">DELETE</button>
                                </form>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>
</body>
</html>


