<?php
require __DIR__ . '/config.php';
require_admin_login();

$generatedKey = '';
$successMessage = '';
$errorMessage = '';

function generate_license_key(mysqli $mysqli): string
{
    // shah + 8 random digits, ensure uniqueness
    while (true) {
        $digits = '';
        for ($i = 0; $i < 8; $i++) {
            $digits .= random_int(0, 9);
        }
        $key = 'shah' . $digits;

        $stmt = $mysqli->prepare('SELECT id FROM licenses WHERE license_key = ? LIMIT 1');
        $stmt->bind_param('s', $key);
        $stmt->execute();
        $stmt->store_result();
        $exists = $stmt->num_rows > 0;
        $stmt->close();

        if (!$exists) {
            return $key;
        }
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $clientName = trim($_POST['client_name'] ?? '');
    $trialDays = (int)($_POST['trial_days'] ?? 0);

    $allowedDays = [3, 5, 7, 14, 30];

    if ($clientName === '') {
        $errorMessage = 'Client name is required.';
    } elseif (!in_array($trialDays, $allowedDays, true)) {
        $errorMessage = 'Invalid trial days selected.';
    } else {
        $generatedKey = generate_license_key($mysqli);

        $expiryDate = (new DateTime())->modify('+' . $trialDays . ' days')->format('Y-m-d');

        $stmt = $mysqli->prepare('INSERT INTO licenses (client_name, license_key, status, expiry_date, created_at) VALUES (?, ?, \'active\', ?, NOW())');
        $stmt->bind_param('sss', $clientName, $generatedKey, $expiryDate);

        if ($stmt->execute()) {
            $successMessage = 'License generated successfully.';
        } else {
            $errorMessage = 'Error saving license: ' . $stmt->error;
            $generatedKey = '';
        }
        $stmt->close();
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Generate License - License Panel</title>
    <link rel="stylesheet" href="styles.css">
</head>
<body>
<div class="container">
    <div class="top-nav">
        <div class="top-nav-title">Generate New License</div>
        <div class="top-nav-actions">
            <a href="dashboard.php" class="btn btn-outline">Back to Dashboard</a>
            <a href="login.php?action=logout" class="btn btn-outline">Logout</a>
        </div>
    </div>

    <div class="card">
        <div class="page-title" style="margin-bottom: 10px;">New License</div>
        <p class="muted" style="margin-bottom: 16px;">Create a new license key for a client with selected trial days.</p>

        <?php if ($errorMessage): ?>
            <div class="alert alert-error"><?php echo h($errorMessage); ?></div>
        <?php endif; ?>
        <?php if ($successMessage): ?>
            <div class="alert alert-success"><?php echo h($successMessage); ?></div>
        <?php endif; ?>

        <form method="post" autocomplete="off">
            <div class="form-group">
                <label class="form-label" for="client_name">Client Name</label>
                <input class="form-control" type="text" id="client_name" name="client_name" required value="<?php echo h($_POST['client_name'] ?? ''); ?>">
            </div>
            <div class="form-group">
                <label class="form-label" for="trial_days">Trial Days</label>
                <select class="form-select" id="trial_days" name="trial_days" required>
                    <option value="">Select days</option>
                    <?php foreach ([3,5,7,14,30] as $d): ?>
                        <option value="<?php echo $d; ?>" <?php echo (isset($_POST['trial_days']) && (int)$_POST['trial_days'] === $d) ? 'selected' : ''; ?>>
                            <?php echo $d; ?> days
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <button type="submit" class="btn btn-primary">Generate</button>
        </form>

        <?php if ($generatedKey): ?>
            <div style="margin-top: 18px;">
                <div class="form-label">Generated License Key</div>
                <div class="generated-key-box">
                    <span class="generated-key" id="generatedKey"><?php echo h($generatedKey); ?></span>
                    <button class="btn btn-outline small" type="button" onclick="copyKey()">Copy</button>
                </div>
                <p class="muted" style="margin-top: 8px;">You can copy this key and send it to the client.</p>
            </div>
        <?php endif; ?>
    </div>
</div>
<script>
    function copyKey() {
        var el = document.getElementById('generatedKey');
        if (!el) return;
        var text = el.textContent || el.innerText;
        if (!navigator.clipboard) {
            var temp = document.createElement('textarea');
            temp.value = text;
            document.body.appendChild(temp);
            temp.select();
            document.execCommand('copy');
            document.body.removeChild(temp);
        } else {
            navigator.clipboard.writeText(text).catch(function () {});
        }
        alert('License key copied!');
    }
</script>
</body>
</html>


